<?php
include('env-helper.php');
include('connect.php');

header('Content-Type: application/json');

// ✅ Get Paystack Secret Key
$paystack_secret = getEnvValue('PAYSTACK_SECRET_KEY');

// ✅ Allowed Paystack IPs for extra security (optional)
$allowed_ips = ['52.31.139.75','52.49.173.169','52.214.14.220'];
$client_ip = $_SERVER['REMOTE_ADDR'] ?? '';

if (!in_array($client_ip, $allowed_ips)) {
    http_response_code(403);
    error_log("Unauthorized IP: " . $client_ip);
    exit(json_encode(['status' => 'error', 'message' => 'Forbidden']));
}

// ✅ Get raw input and signature
$input = @file_get_contents("php://input");
$signature = $_SERVER['HTTP_X_PAYSTACK_SIGNATURE'] ?? '';

if (!$signature) {
    http_response_code(400);
    error_log("Missing Paystack signature");
    exit(json_encode(['status' => 'error', 'message' => 'Missing signature']));
}

// ✅ Verify signature
$computed_signature = hash_hmac('sha512', $input, $paystack_secret);
if (!hash_equals($signature, $computed_signature)) {
    http_response_code(400);
    error_log("Invalid Paystack signature");
    exit(json_encode(['status' => 'error', 'message' => 'Invalid signature']));
}

// ✅ Decode JSON payload
$event = json_decode($input, true);
if (!$event || !isset($event['event'], $event['data'])) {
    http_response_code(400);
    error_log("Invalid JSON or missing event/data");
    exit(json_encode(['status' => 'error', 'message' => 'Invalid payload']));
}

if ($event['event'] === 'charge.success') {
    $data = $event['data'];
    $reference = $data['reference'];
    $amount = $data['amount'] / 100; // Paystack sends amount in kobo
    $status = $data['status'];
    $paid_at = strtotime($data['paid_at']);

    // ✅ Reject old events (older than 30 mins)
    if (time() - $paid_at > 1800) {
        http_response_code(400);
        error_log("Old event rejected: Ref $reference");
        exit(json_encode(['status' => 'error', 'message' => 'Old event']));
    }

    try {
        // ✅ Start transaction
        $conn->begin_transaction();

        // ✅ Check if reference exists in deposits table
        $stmt = $conn->prepare("SELECT status, user_id FROM deposits WHERE reference = ?");
        $stmt->bind_param("s", $reference);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            // ✅ Insert new deposit record
            $user_id = $data['metadata']['user_id'] ?? null;
            if (!$user_id) {
                throw new Exception("Missing user_id in metadata for ref $reference");
            }

            $insert = $conn->prepare("INSERT INTO deposits (user_id, amount, reference, status) VALUES (?, ?, ?, 'pending')");
            $insert->bind_param("ids", $user_id, $amount, $reference);
            $insert->execute();
        } else {
            $row = $result->fetch_assoc();
            if ($row['status'] === 'approved') {
                // ✅ Already processed → exit quietly
                http_response_code(200);
                exit(json_encode(['status' => 'ok', 'message' => 'Already processed']));
            }
            $user_id = $row['user_id'];
        }

        // ✅ Update deposit status
        $update = $conn->prepare("UPDATE deposits SET status = 'approved' WHERE reference = ?");
        $update->bind_param("s", $reference);
        $update->execute();

        // ✅ Credit user wallet
        $wallet = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
        $wallet->bind_param("di", $amount, $user_id);
        $wallet->execute();

        $conn->commit();

        // ✅ Log success
        error_log("Payment successful: Ref $reference | User $user_id | Amount $amount");

        http_response_code(200);
        echo json_encode(['status' => 'success']);
    } catch (Exception $e) {
        $conn->rollback();
        error_log("Webhook error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Internal Server Error']);
    }
} else {
    http_response_code(200);
    echo json_encode(['status' => 'ignored']);
}
?>
